// Main Application Controller

class Application {
    constructor() {
        this.currentPage = 'dashboard';
        this.pageManagers = {
            dashboard: window.dashboard,
            contracts: null, // Will be loaded when needed
            scenes: null,
            reports: null,
            users: null,
            settings: null,
            backup: null
        };
        this.sidebarCollapsed = false;
        this.isMobile = window.innerWidth <= 768;
    }

    async init() {
        this.setupEventListeners();
        this.setupResponsiveHandling();
        this.initializeTooltipsAndPopovers();
        
        // Navigate to dashboard by default
        await this.navigateTo('dashboard');
    }

    setupEventListeners() {
        // Sidebar toggle
        const sidebarToggle = document.getElementById('sidebarToggle');
        sidebarToggle.addEventListener('click', () => {
            this.toggleSidebar();
        });

        // Menu navigation
        const menuItems = document.querySelectorAll('.menu-item');
        menuItems.forEach(item => {
            item.addEventListener('click', (e) => {
                e.preventDefault();
                const page = item.dataset.page;
                if (page) {
                    this.navigateTo(page);
                }
            });
        });

        // Handle browser back/forward
        window.addEventListener('popstate', (e) => {
            if (e.state && e.state.page) {
                this.navigateTo(e.state.page, e.state.params, false);
            }
        });

        // Handle window resize
        window.addEventListener('resize', () => {
            this.handleResize();
        });

        // Handle clicks outside sidebar on mobile
        document.addEventListener('click', (e) => {
            if (this.isMobile && !e.target.closest('#sidebar') && !e.target.closest('#sidebarToggle')) {
                this.hideSidebarOnMobile();
            }
        });
    }

    setupResponsiveHandling() {
        this.handleResize();
    }

    handleResize() {
        const wasMobile = this.isMobile;
        this.isMobile = window.innerWidth <= 768;

        if (wasMobile !== this.isMobile) {
            if (this.isMobile) {
                this.hideSidebarOnMobile();
            } else {
                this.showSidebarOnDesktop();
            }
        }
    }

    toggleSidebar() {
        if (this.isMobile) {
            this.toggleSidebarOnMobile();
        } else {
            this.toggleSidebarOnDesktop();
        }
    }

    toggleSidebarOnMobile() {
        const sidebar = document.getElementById('sidebar');
        sidebar.classList.toggle('show');
    }

    hideSidebarOnMobile() {
        const sidebar = document.getElementById('sidebar');
        sidebar.classList.remove('show');
    }

    toggleSidebarOnDesktop() {
        const sidebar = document.getElementById('sidebar');
        sidebar.classList.toggle('collapsed');
        this.sidebarCollapsed = sidebar.classList.contains('collapsed');
    }

    showSidebarOnDesktop() {
        const sidebar = document.getElementById('sidebar');
        sidebar.classList.remove('show', 'collapsed');
        this.sidebarCollapsed = false;
    }

    initializeTooltipsAndPopovers() {
        utils.initializeTooltips();
        utils.initializePopovers();
    }

    async navigateTo(page, params = {}, updateHistory = true) {
        // Check permissions
        if (!this.checkPagePermissions(page)) {
            return;
        }

        // Update URL and history
        if (updateHistory) {
            const url = new URL(window.location);
            url.searchParams.set('page', page);
            if (Object.keys(params).length > 0) {
                url.searchParams.set('params', JSON.stringify(params));
            } else {
                url.searchParams.delete('params');
            }
            
            history.pushState({ page, params }, '', url);
        }

        // Update active menu item
        this.updateActiveMenuItem(page);

        // Hide sidebar on mobile after navigation
        if (this.isMobile) {
            this.hideSidebarOnMobile();
        }

        // Clean up current page
        if (this.pageManagers[this.currentPage] && this.pageManagers[this.currentPage].destroy) {
            this.pageManagers[this.currentPage].destroy();
        }

        // Load and render new page
        try {
            await this.loadPage(page, params);
            this.currentPage = page;
        } catch (error) {
            console.error(`Error loading page ${page}:`, error);
            utils.showToast(`خطأ في تحميل الصفحة: ${error.message}`, 'danger');
        }
    }

    checkPagePermissions(page) {
        // Pages that require admin access
        const adminPages = ['users', 'settings', 'backup'];
        
        if (adminPages.includes(page) && !auth.isAdmin()) {
            utils.showToast('هذه الصفحة تتطلب صلاحيات المدير', 'warning');
            return false;
        }

        // Check specific permissions for other pages
        switch (page) {
            case 'contracts':
                return auth.hasPermission('contracts', 'view');
            case 'scenes':
                return auth.hasPermission('scenes', 'view');
            case 'reports':
                return auth.hasPermission('contracts', 'print') || auth.hasPermission('scenes', 'print');
            default:
                return true;
        }
    }

    updateActiveMenuItem(page) {
        // Remove active class from all menu items
        document.querySelectorAll('.menu-item').forEach(item => {
            item.classList.remove('active');
        });

        // Add active class to current page menu item
        const activeItem = document.querySelector(`.menu-item[data-page="${page}"]`);
        if (activeItem) {
            activeItem.classList.add('active');
        }
    }

    async loadPage(page, params = {}) {
        // Load page manager if not already loaded
        if (!this.pageManagers[page]) {
            await this.loadPageManager(page);
        }

        // Render the page
        if (this.pageManagers[page] && this.pageManagers[page].render) {
            await this.pageManagers[page].render(params);
        } else {
            this.renderNotFound(page);
        }

        // Re-initialize tooltips and popovers for new content
        setTimeout(() => {
            this.initializeTooltipsAndPopovers();
        }, 100);
    }

    async loadPageManager(page) {
        try {
            switch (page) {
                case 'dashboard':
                    this.pageManagers[page] = window.dashboard;
                    break;
                case 'contracts':
                    if (!window.contracts) {
                        await this.loadScript('js/contracts.js');
                    }
                    this.pageManagers[page] = window.contracts;
                    break;
                case 'scenes':
                    if (!window.scenes) {
                        await this.loadScript('js/scenes.js');
                    }
                    this.pageManagers[page] = window.scenes;
                    break;
                case 'reports':
                    if (!window.reports) {
                        await this.loadScript('js/reports.js');
                    }
                    this.pageManagers[page] = window.reports;
                    break;
                case 'users':
                    if (!window.users) {
                        await this.loadScript('js/users.js');
                    }
                    this.pageManagers[page] = window.users;
                    break;
                case 'settings':
                    if (!window.settings) {
                        await this.loadScript('js/settings.js');
                    }
                    this.pageManagers[page] = window.settings;
                    break;
                case 'backup':
                    if (!window.backup) {
                        await this.loadScript('js/backup.js');
                    }
                    this.pageManagers[page] = window.backup;
                    break;
                default:
                    throw new Error(`Unknown page: ${page}`);
            }
        } catch (error) {
            console.error(`Error loading page manager for ${page}:`, error);
            throw error;
        }
    }

    loadScript(src) {
        return new Promise((resolve, reject) => {
            const script = document.createElement('script');
            script.src = src;
            script.onload = resolve;
            script.onerror = reject;
            document.head.appendChild(script);
        });
    }

    renderNotFound(page) {
        const content = `
            <div class="text-center py-5">
                <i class="fas fa-exclamation-triangle fa-5x text-warning mb-4"></i>
                <h2>الصفحة غير موجودة</h2>
                <p class="text-muted">الصفحة المطلوبة "${page}" غير موجودة أو لا يمكن الوصول إليها.</p>
                <button class="btn btn-primary" onclick="app.navigateTo('dashboard')">
                    <i class="fas fa-home me-2"></i>
                    العودة للرئيسية
                </button>
            </div>
        `;
        
        document.getElementById('pageContent').innerHTML = content;
    }

    // Utility method to get current page parameters
    getCurrentParams() {
        const urlParams = new URLSearchParams(window.location.search);
        const paramsStr = urlParams.get('params');
        return paramsStr ? JSON.parse(paramsStr) : {};
    }

    // Utility method to refresh current page
    async refreshCurrentPage() {
        const params = this.getCurrentParams();
        await this.navigateTo(this.currentPage, params, false);
    }

    // Show global loading state
    showGlobalLoading() {
        utils.showLoading();
    }

    // Hide global loading state
    hideGlobalLoading() {
        utils.hideLoading();
    }

    // Handle global errors
    handleGlobalError(error) {
        console.error('Global error:', error);
        utils.showToast('حدث خطأ غير متوقع', 'danger');
    }
}

// Initialize application when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    // Create global app instance
    window.app = new Application();
    
    // The app will be initialized by the auth manager after successful login
});

// Handle global errors
window.addEventListener('error', (e) => {
    if (window.app) {
        window.app.handleGlobalError(e.error);
    }
});

window.addEventListener('unhandledrejection', (e) => {
    if (window.app) {
        window.app.handleGlobalError(e.reason);
    }
});
