const express = require('express');
const { query, validationResult } = require('express-validator');
const { Contract, Scene, User, Settings } = require('../models');
const { authenticateToken, checkPermission } = require('../middleware/auth');
const { Op } = require('sequelize');
const puppeteer = require('puppeteer');
const ExcelJS = require('exceljs');
const path = require('path');
const fs = require('fs').promises;
const moment = require('moment');

const router = express.Router();

// Helper function to generate PDF
async function generatePDF(htmlContent, options = {}) {
  const browser = await puppeteer.launch({
    headless: true,
    args: ['--no-sandbox', '--disable-setuid-sandbox']
  });
  
  const page = await browser.newPage();
  await page.setContent(htmlContent, { waitUntil: 'networkidle0' });
  
  const pdfOptions = {
    format: 'A4',
    printBackground: true,
    margin: {
      top: options.marginTop || '20mm',
      bottom: options.marginBottom || '20mm',
      left: options.marginLeft || '15mm',
      right: options.marginRight || '15mm'
    }
  };
  
  const pdf = await page.pdf(pdfOptions);
  await browser.close();
  
  return pdf;
}

// Helper function to generate HTML template
function generateHTMLTemplate(title, content, header = '', footer = '') {
  return `
    <!DOCTYPE html>
    <html dir="rtl" lang="ar">
    <head>
      <meta charset="UTF-8">
      <meta name="viewport" content="width=device-width, initial-scale=1.0">
      <title>${title}</title>
      <style>
        body {
          font-family: 'Arial', sans-serif;
          direction: rtl;
          text-align: right;
          margin: 0;
          padding: 20px;
          font-size: 12px;
          line-height: 1.6;
        }
        .header {
          text-align: center;
          border-bottom: 2px solid #333;
          padding-bottom: 10px;
          margin-bottom: 20px;
        }
        .footer {
          text-align: center;
          border-top: 1px solid #ccc;
          padding-top: 10px;
          margin-top: 20px;
          font-size: 10px;
        }
        .title {
          font-size: 18px;
          font-weight: bold;
          text-align: center;
          margin: 20px 0;
        }
        table {
          width: 100%;
          border-collapse: collapse;
          margin: 10px 0;
        }
        th, td {
          border: 1px solid #ddd;
          padding: 8px;
          text-align: right;
        }
        th {
          background-color: #f5f5f5;
          font-weight: bold;
        }
        .info-section {
          margin: 15px 0;
          padding: 10px;
          background-color: #f9f9f9;
          border-radius: 5px;
        }
        .info-row {
          display: flex;
          justify-content: space-between;
          margin: 5px 0;
        }
        .label {
          font-weight: bold;
          min-width: 120px;
        }
        @media print {
          body { margin: 0; }
          .page-break { page-break-before: always; }
        }
      </style>
    </head>
    <body>
      ${header ? `<div class="header">${header}</div>` : ''}
      <div class="title">${title}</div>
      ${content}
      ${footer ? `<div class="footer">${footer}</div>` : ''}
    </body>
    </html>
  `;
}

// Get dashboard statistics
router.get('/dashboard', [
  authenticateToken,
  checkPermission('contracts', 'view')
], async (req, res) => {
  try {
    const today = new Date();
    const startOfMonth = new Date(today.getFullYear(), today.getMonth(), 1);
    const startOfYear = new Date(today.getFullYear(), 0, 1);

    // Contracts statistics
    const totalContracts = await Contract.count();
    const activeContracts = await Contract.count({ where: { status: 'active' } });
    const expiredContracts = await Contract.count({ where: { status: 'expired' } });
    const monthlyContracts = await Contract.count({
      where: {
        createdAt: { [Op.gte]: startOfMonth }
      }
    });

    // Scenes statistics
    const totalScenes = await Scene.count();
    const pendingScenes = await Scene.count({ where: { status: 'pending' } });
    const completedScenes = await Scene.count({ where: { status: 'completed' } });
    const monthlyScenes = await Scene.count({
      where: {
        createdAt: { [Op.gte]: startOfMonth }
      }
    });

    // Recent activities
    const recentContracts = await Contract.findAll({
      limit: 5,
      order: [['createdAt', 'DESC']],
      attributes: ['id', 'contractNumber', 'secondParty', 'createdAt'],
      include: [
        {
          model: User,
          as: 'creator',
          attributes: ['fullName']
        }
      ]
    });

    const recentScenes = await Scene.findAll({
      limit: 5,
      order: [['createdAt', 'DESC']],
      attributes: ['id', 'date', 'action', 'applicantName', 'createdAt'],
      include: [
        {
          model: Contract,
          as: 'contract',
          attributes: ['contractNumber']
        }
      ]
    });

    res.json({
      success: true,
      data: {
        contracts: {
          total: totalContracts,
          active: activeContracts,
          expired: expiredContracts,
          monthly: monthlyContracts
        },
        scenes: {
          total: totalScenes,
          pending: pendingScenes,
          completed: completedScenes,
          monthly: monthlyScenes
        },
        recent: {
          contracts: recentContracts,
          scenes: recentScenes
        }
      }
    });
  } catch (error) {
    console.error('Dashboard statistics error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب إحصائيات لوحة التحكم'
    });
  }
});

// Generate contract report
router.get('/contract/:id', [
  authenticateToken,
  checkPermission('contracts', 'print'),
  query('includeScenes').optional().isBoolean(),
  query('format').optional().isIn(['pdf', 'excel'])
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const includeScenes = req.query.includeScenes === 'true';
    const format = req.query.format || 'pdf';

    const contract = await Contract.findByPk(req.params.id, {
      include: [
        {
          model: User,
          as: 'creator',
          attributes: ['fullName']
        },
        ...(includeScenes ? [{
          model: Scene,
          as: 'scenes',
          include: [
            {
              model: User,
              as: 'creator',
              attributes: ['fullName']
            }
          ],
          order: [['date', 'DESC']]
        }] : [])
      ]
    });

    if (!contract) {
      return res.status(404).json({
        success: false,
        message: 'العقد غير موجود'
      });
    }

    if (format === 'excel') {
      // Generate Excel report
      const workbook = new ExcelJS.Workbook();
      const worksheet = workbook.addWorksheet('تقرير العقد');

      // Set RTL direction
      worksheet.views = [{ rightToLeft: true }];

      // Add headers
      worksheet.addRow(['تقرير العقد رقم: ' + contract.contractNumber]);
      worksheet.addRow([]);

      // Contract details
      worksheet.addRow(['تفاصيل العقد']);
      worksheet.addRow(['رقم العقد', contract.contractNumber]);
      worksheet.addRow(['تاريخ العقد', moment(contract.contractDate).format('YYYY-MM-DD')]);
      worksheet.addRow(['الطرف الثاني', contract.secondParty]);
      worksheet.addRow(['الموقع', contract.location]);
      worksheet.addRow(['اسم المالك', contract.ownerName]);
      worksheet.addRow(['رقم الهوية', contract.ownerNationalID]);
      worksheet.addRow(['رقم الهاتف', contract.ownerPhoneNumber]);
      worksheet.addRow(['نوع الكاميرات', contract.camerasType]);
      worksheet.addRow(['دقة الكاميرات', contract.camerasResolution]);
      worksheet.addRow(['عدد الكاميرات الداخلية', contract.innerCamerasNumber]);
      worksheet.addRow(['عدد الكاميرات الخارجية', contract.outerCamerasNumber]);
      worksheet.addRow(['سعة التخزين', contract.storageCapacity]);
      worksheet.addRow(['تاريخ البداية', moment(contract.startsFrom).format('YYYY-MM-DD')]);
      worksheet.addRow(['تاريخ النهاية', moment(contract.endsAt).format('YYYY-MM-DD')]);
      worksheet.addRow(['الحالة', contract.status]);

      if (includeScenes && contract.scenes && contract.scenes.length > 0) {
        worksheet.addRow([]);
        worksheet.addRow(['المشاهد المرتبطة']);
        worksheet.addRow(['التاريخ', 'نوع العمل', 'مقدم الطلب', 'الهاتف', 'الحالة', 'الفني']);

        contract.scenes.forEach(scene => {
          worksheet.addRow([
            moment(scene.date).format('YYYY-MM-DD'),
            scene.action,
            scene.applicantName,
            scene.applicantPhone,
            scene.status,
            scene.technician || '-'
          ]);
        });
      }

      res.setHeader('Content-Type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
      res.setHeader('Content-Disposition', `attachment; filename="contract-${contract.contractNumber}.xlsx"`);

      await workbook.xlsx.write(res);
      res.end();
    } else {
      // Generate PDF report
      const header = await Settings.getValue('print_header', 'نظام إدارة عقود الصيانة');
      const footer = await Settings.getValue('print_footer', 'جميع الحقوق محفوظة');

      let content = `
        <div class="info-section">
          <h3>تفاصيل العقد</h3>
          <div class="info-row"><span class="label">رقم العقد:</span> ${contract.contractNumber}</div>
          <div class="info-row"><span class="label">تاريخ العقد:</span> ${moment(contract.contractDate).format('YYYY-MM-DD')}</div>
          <div class="info-row"><span class="label">الطرف الثاني:</span> ${contract.secondParty}</div>
          <div class="info-row"><span class="label">الموقع:</span> ${contract.location}</div>
          <div class="info-row"><span class="label">اسم المالك:</span> ${contract.ownerName}</div>
          <div class="info-row"><span class="label">رقم الهوية:</span> ${contract.ownerNationalID}</div>
          <div class="info-row"><span class="label">رقم الهاتف:</span> ${contract.ownerPhoneNumber}</div>
          <div class="info-row"><span class="label">نوع الكاميرات:</span> ${contract.camerasType}</div>
          <div class="info-row"><span class="label">دقة الكاميرات:</span> ${contract.camerasResolution}</div>
          <div class="info-row"><span class="label">عدد الكاميرات الداخلية:</span> ${contract.innerCamerasNumber}</div>
          <div class="info-row"><span class="label">عدد الكاميرات الخارجية:</span> ${contract.outerCamerasNumber}</div>
          <div class="info-row"><span class="label">سعة التخزين:</span> ${contract.storageCapacity}</div>
          <div class="info-row"><span class="label">تاريخ البداية:</span> ${moment(contract.startsFrom).format('YYYY-MM-DD')}</div>
          <div class="info-row"><span class="label">تاريخ النهاية:</span> ${moment(contract.endsAt).format('YYYY-MM-DD')}</div>
          <div class="info-row"><span class="label">الحالة:</span> ${contract.status}</div>
        </div>
      `;

      if (includeScenes && contract.scenes && contract.scenes.length > 0) {
        content += `
          <div class="page-break"></div>
          <h3>المشاهد المرتبطة</h3>
          <table>
            <thead>
              <tr>
                <th>التاريخ</th>
                <th>نوع العمل</th>
                <th>مقدم الطلب</th>
                <th>الهاتف</th>
                <th>الحالة</th>
                <th>الفني</th>
              </tr>
            </thead>
            <tbody>
        `;

        contract.scenes.forEach(scene => {
          content += `
            <tr>
              <td>${moment(scene.date).format('YYYY-MM-DD')}</td>
              <td>${scene.action}</td>
              <td>${scene.applicantName}</td>
              <td>${scene.applicantPhone}</td>
              <td>${scene.status}</td>
              <td>${scene.technician || '-'}</td>
            </tr>
          `;
        });

        content += '</tbody></table>';
      }

      const html = generateHTMLTemplate(`تقرير العقد رقم: ${contract.contractNumber}`, content, header, footer);
      const pdf = await generatePDF(html);

      res.setHeader('Content-Type', 'application/pdf');
      res.setHeader('Content-Disposition', `attachment; filename="contract-${contract.contractNumber}.pdf"`);
      res.send(pdf);
    }
  } catch (error) {
    console.error('Generate contract report error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في إنشاء تقرير العقد'
    });
  }
});

// Generate scene report
router.get('/scene/:id', [
  authenticateToken,
  checkPermission('scenes', 'print'),
  query('format').optional().isIn(['pdf', 'excel'])
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const format = req.query.format || 'pdf';

    const scene = await Scene.findByPk(req.params.id, {
      include: [
        {
          model: Contract,
          as: 'contract'
        },
        {
          model: User,
          as: 'creator',
          attributes: ['fullName']
        }
      ]
    });

    if (!scene) {
      return res.status(404).json({
        success: false,
        message: 'المشهد غير موجود'
      });
    }

    if (format === 'excel') {
      // Generate Excel report
      const workbook = new ExcelJS.Workbook();
      const worksheet = workbook.addWorksheet('تقرير المشهد');

      worksheet.views = [{ rightToLeft: true }];

      worksheet.addRow(['تقرير المشهد رقم: ' + scene.id]);
      worksheet.addRow([]);

      // Scene details
      worksheet.addRow(['تفاصيل المشهد']);
      worksheet.addRow(['رقم المشهد', scene.id]);
      worksheet.addRow(['رقم العقد', scene.contract.contractNumber]);
      worksheet.addRow(['الطرف الثاني', scene.contract.secondParty]);
      worksheet.addRow(['التاريخ', moment(scene.date).format('YYYY-MM-DD')]);
      worksheet.addRow(['نوع العمل', scene.action]);
      worksheet.addRow(['مقدم الطلب', scene.applicantName]);
      worksheet.addRow(['هاتف مقدم الطلب', scene.applicantPhone]);
      worksheet.addRow(['الفني', scene.technician || '-']);
      worksheet.addRow(['وقت البداية', scene.startTime || '-']);
      worksheet.addRow(['وقت النهاية', scene.endTime || '-']);
      worksheet.addRow(['الحالة', scene.status]);
      worksheet.addRow(['وصف العمل', scene.workDescription || '-']);
      worksheet.addRow(['الملاحظات', scene.notes || '-']);

      res.setHeader('Content-Type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
      res.setHeader('Content-Disposition', `attachment; filename="scene-${scene.id}.xlsx"`);

      await workbook.xlsx.write(res);
      res.end();
    } else {
      // Generate PDF report
      const header = await Settings.getValue('print_header', 'نظام إدارة عقود الصيانة');
      const footer = await Settings.getValue('print_footer', 'جميع الحقوق محفوظة');

      const content = `
        <div class="info-section">
          <h3>تفاصيل المشهد</h3>
          <div class="info-row"><span class="label">رقم المشهد:</span> ${scene.id}</div>
          <div class="info-row"><span class="label">رقم العقد:</span> ${scene.contract.contractNumber}</div>
          <div class="info-row"><span class="label">الطرف الثاني:</span> ${scene.contract.secondParty}</div>
          <div class="info-row"><span class="label">التاريخ:</span> ${moment(scene.date).format('YYYY-MM-DD')}</div>
          <div class="info-row"><span class="label">نوع العمل:</span> ${scene.action}</div>
          <div class="info-row"><span class="label">مقدم الطلب:</span> ${scene.applicantName}</div>
          <div class="info-row"><span class="label">هاتف مقدم الطلب:</span> ${scene.applicantPhone}</div>
          <div class="info-row"><span class="label">الفني:</span> ${scene.technician || '-'}</div>
          <div class="info-row"><span class="label">وقت البداية:</span> ${scene.startTime || '-'}</div>
          <div class="info-row"><span class="label">وقت النهاية:</span> ${scene.endTime || '-'}</div>
          <div class="info-row"><span class="label">الحالة:</span> ${scene.status}</div>
          <div class="info-row"><span class="label">وصف العمل:</span> ${scene.workDescription || '-'}</div>
          <div class="info-row"><span class="label">الملاحظات:</span> ${scene.notes || '-'}</div>
        </div>
      `;

      const html = generateHTMLTemplate(`تقرير المشهد رقم: ${scene.id}`, content, header, footer);
      const pdf = await generatePDF(html);

      res.setHeader('Content-Type', 'application/pdf');
      res.setHeader('Content-Disposition', `attachment; filename="scene-${scene.id}.pdf"`);
      res.send(pdf);
    }
  } catch (error) {
    console.error('Generate scene report error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في إنشاء تقرير المشهد'
    });
  }
});

// Generate time-based reports
router.get('/time-range', [
  authenticateToken,
  checkPermission('contracts', 'print'),
  query('type').isIn(['contracts', 'scenes']).withMessage('نوع التقرير مطلوب'),
  query('period').optional().isIn(['daily', 'weekly', 'monthly', 'quarterly', 'yearly', 'custom']),
  query('dateFrom').optional().isISO8601(),
  query('dateTo').optional().isISO8601(),
  query('format').optional().isIn(['pdf', 'excel'])
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const { type, period = 'custom', format = 'pdf' } = req.query;
    let { dateFrom, dateTo } = req.query;

    // Calculate date range based on period
    const today = new Date();
    switch (period) {
      case 'daily':
        dateFrom = moment().format('YYYY-MM-DD');
        dateTo = moment().format('YYYY-MM-DD');
        break;
      case 'weekly':
        dateFrom = moment().startOf('week').format('YYYY-MM-DD');
        dateTo = moment().endOf('week').format('YYYY-MM-DD');
        break;
      case 'monthly':
        dateFrom = moment().startOf('month').format('YYYY-MM-DD');
        dateTo = moment().endOf('month').format('YYYY-MM-DD');
        break;
      case 'quarterly':
        dateFrom = moment().startOf('quarter').format('YYYY-MM-DD');
        dateTo = moment().endOf('quarter').format('YYYY-MM-DD');
        break;
      case 'yearly':
        dateFrom = moment().startOf('year').format('YYYY-MM-DD');
        dateTo = moment().endOf('year').format('YYYY-MM-DD');
        break;
    }

    if (!dateFrom || !dateTo) {
      return res.status(400).json({
        success: false,
        message: 'تاريخ البداية والنهاية مطلوبان'
      });
    }

    let data, title;
    const whereClause = {
      [type === 'contracts' ? 'contractDate' : 'date']: {
        [Op.between]: [dateFrom, dateTo]
      }
    };

    if (type === 'contracts') {
      data = await Contract.findAll({
        where: whereClause,
        include: [
          {
            model: User,
            as: 'creator',
            attributes: ['fullName']
          }
        ],
        order: [['contractDate', 'DESC']]
      });
      title = `تقرير العقود من ${dateFrom} إلى ${dateTo}`;
    } else {
      data = await Scene.findAll({
        where: whereClause,
        include: [
          {
            model: Contract,
            as: 'contract',
            attributes: ['contractNumber', 'secondParty']
          },
          {
            model: User,
            as: 'creator',
            attributes: ['fullName']
          }
        ],
        order: [['date', 'DESC']]
      });
      title = `تقرير المشاهد من ${dateFrom} إلى ${dateTo}`;
    }

    if (format === 'excel') {
      const workbook = new ExcelJS.Workbook();
      const worksheet = workbook.addWorksheet(type === 'contracts' ? 'تقرير العقود' : 'تقرير المشاهد');
      worksheet.views = [{ rightToLeft: true }];

      worksheet.addRow([title]);
      worksheet.addRow([]);

      if (type === 'contracts') {
        worksheet.addRow(['رقم العقد', 'تاريخ العقد', 'الطرف الثاني', 'الموقع', 'اسم المالك', 'الحالة']);
        data.forEach(contract => {
          worksheet.addRow([
            contract.contractNumber,
            moment(contract.contractDate).format('YYYY-MM-DD'),
            contract.secondParty,
            contract.location,
            contract.ownerName,
            contract.status
          ]);
        });
      } else {
        worksheet.addRow(['رقم العقد', 'التاريخ', 'نوع العمل', 'مقدم الطلب', 'الهاتف', 'الحالة']);
        data.forEach(scene => {
          worksheet.addRow([
            scene.contract.contractNumber,
            moment(scene.date).format('YYYY-MM-DD'),
            scene.action,
            scene.applicantName,
            scene.applicantPhone,
            scene.status
          ]);
        });
      }

      res.setHeader('Content-Type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
      res.setHeader('Content-Disposition', `attachment; filename="${type}-report-${dateFrom}-${dateTo}.xlsx"`);

      await workbook.xlsx.write(res);
      res.end();
    } else {
      const header = await Settings.getValue('print_header', 'نظام إدارة عقود الصيانة');
      const footer = await Settings.getValue('print_footer', 'جميع الحقوق محفوظة');

      let content = '<table><thead><tr>';

      if (type === 'contracts') {
        content += '<th>رقم العقد</th><th>تاريخ العقد</th><th>الطرف الثاني</th><th>الموقع</th><th>اسم المالك</th><th>الحالة</th>';
        content += '</tr></thead><tbody>';
        
        data.forEach(contract => {
          content += `
            <tr>
              <td>${contract.contractNumber}</td>
              <td>${moment(contract.contractDate).format('YYYY-MM-DD')}</td>
              <td>${contract.secondParty}</td>
              <td>${contract.location}</td>
              <td>${contract.ownerName}</td>
              <td>${contract.status}</td>
            </tr>
          `;
        });
      } else {
        content += '<th>رقم العقد</th><th>التاريخ</th><th>نوع العمل</th><th>مقدم الطلب</th><th>الهاتف</th><th>الحالة</th>';
        content += '</tr></thead><tbody>';
        
        data.forEach(scene => {
          content += `
            <tr>
              <td>${scene.contract.contractNumber}</td>
              <td>${moment(scene.date).format('YYYY-MM-DD')}</td>
              <td>${scene.action}</td>
              <td>${scene.applicantName}</td>
              <td>${scene.applicantPhone}</td>
              <td>${scene.status}</td>
            </tr>
          `;
        });
      }

      content += '</tbody></table>';
      content += `<p>إجمالي العدد: ${data.length}</p>`;

      const html = generateHTMLTemplate(title, content, header, footer);
      const pdf = await generatePDF(html);

      res.setHeader('Content-Type', 'application/pdf');
      res.setHeader('Content-Disposition', `attachment; filename="${type}-report-${dateFrom}-${dateTo}.pdf"`);
      res.send(pdf);
    }
  } catch (error) {
    console.error('Generate time range report error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في إنشاء التقرير'
    });
  }
});

module.exports = router;
