const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Scene = sequelize.define('Scene', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  contractId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    references: {
      model: 'contracts',
      key: 'id'
    }
  },
  facilityName: {
    type: DataTypes.STRING(200),
    allowNull: false,
    validate: {
      len: [2, 200],
      notEmpty: true
    }
  },
  facilityOwnerName: {
    type: DataTypes.STRING(100),
    allowNull: false,
    validate: {
      len: [2, 100],
      notEmpty: true
    }
  },
  ownerNationalID: {
    type: DataTypes.STRING(20),
    allowNull: false,
    validate: {
      len: [10, 20],
      isNumeric: true
    }
  },
  facilityLocation: {
    type: DataTypes.TEXT,
    allowNull: false,
    validate: {
      notEmpty: true
    }
  },
  internalCameras: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0,
    validate: {
      min: 0
    }
  },
  externalCameras: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0,
    validate: {
      min: 0
    }
  },
  date: {
    type: DataTypes.DATEONLY,
    allowNull: false,
    validate: {
      isDate: true
    }
  },
  action: {
    type: DataTypes.STRING(100),
    allowNull: false,
    validate: {
      notEmpty: true,
      isIn: [['صيانة', 'تركيب', 'فحص', 'إصلاح', 'استبدال', 'تحديث', 'أخرى']]
    }
  },
  applicantName: {
    type: DataTypes.STRING(100),
    allowNull: false,
    validate: {
      len: [2, 100],
      notEmpty: true
    }
  },
  applicantPhone: {
    type: DataTypes.STRING(20),
    allowNull: false,
    validate: {
      len: [10, 20],
      isNumeric: true
    }
  },
  attachments: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: []
  },
  notes: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  status: {
    type: DataTypes.ENUM('pending', 'in_progress', 'completed', 'cancelled'),
    defaultValue: 'pending'
  },
  // materialsUsed: {
  //   type: DataTypes.JSON,
  //   allowNull: true,
  //   defaultValue: []
  // },
  createdBy: {
    type: DataTypes.INTEGER,
    allowNull: false,
    references: {
      model: 'users',
      key: 'id'
    }
  },
  updatedBy: {
    type: DataTypes.INTEGER,
    allowNull: true,
    references: {
      model: 'users',
      key: 'id'
    }
  }
}, {
  tableName: 'scenes',
  timestamps: true,
  indexes: [
    {
      fields: ['contractId']
    },
    {
      fields: ['date']
    },
    {
      fields: ['action']
    },
    {
      fields: ['status']
    },
    {
      fields: ['applicantName']
    },
    {
      fields: ['facilityName']
    },
    {
      fields: ['facilityOwnerName']
    },
    {
      fields: ['ownerNationalID']
    }
  ]
});

// Virtual field for total cameras
Scene.prototype.getTotalCameras = function() {
  return this.internalCameras + this.externalCameras;
};

module.exports = Scene;
